﻿Imports System.Windows.Forms.DataVisualization.Charting

Public Class Form1

    Private Const PHY_CYCLE As Integer = 23
    Private Const EMO_CYCLE As Integer = 28
    Private Const INT_CYCLE As Integer = 33

    Private Sub btnGenerate_Click(sender As Object, e As EventArgs) Handles btnGenerate.Click

        Dim birth As Date = dtpBirthdate.Value
        Dim target As Date = dtpTargetDate.Value

        Dim daysPast As Integer = (target - birth).Days

        ' Setup chart
        SetupChart()

        ' Plot 15 days before to 15 days after target date
        Dim startDate As Date = target.AddDays(-15)
        Dim endDate As Date = target.AddDays(15)

        Dim d As Date = startDate
        While d <= endDate

            Dim x As Double = (d - birth).Days

            Dim physical = Math.Sin(2 * Math.PI * x / PHY_CYCLE)
            Dim emotional = Math.Sin(2 * Math.PI * x / EMO_CYCLE)
            Dim intellectual = Math.Sin(2 * Math.PI * x / INT_CYCLE)

            chartBiorhythm.Series("Physical").Points.AddXY(d, physical)
            chartBiorhythm.Series("Emotional").Points.AddXY(d, emotional)
            chartBiorhythm.Series("Intellectual").Points.AddXY(d, intellectual)

            d = d.AddDays(1)
        End While

        ' Add vertical marker at selected date
        AddTargetMarker(target)

        ' Compute today's values
        Dim pToday = Math.Sin(2 * Math.PI * daysPast / PHY_CYCLE)
        Dim eToday = Math.Sin(2 * Math.PI * daysPast / EMO_CYCLE)
        Dim iToday = Math.Sin(2 * Math.PI * daysPast / INT_CYCLE)

        Dim criticalList As String = ""
        If Math.Abs(pToday) < 0.05 Then criticalList &= "Physical "
        If Math.Abs(eToday) < 0.05 Then criticalList &= "Emotional "
        If Math.Abs(iToday) < 0.05 Then criticalList &= "Intellectual "

        lblOutput.Text =
            $"Date: {target.ToShortDateString()}" & vbCrLf &
            $"Physical: {Math.Round(pToday, 3)}" & vbCrLf &
            $"Emotional: {Math.Round(eToday, 3)}" & vbCrLf &
            $"Intellectual: {Math.Round(iToday, 3)}" & vbCrLf &
            If(criticalList <> "", $"CRITICAL: {criticalList}", "No critical cycle today")

        BuildMonthCalendar()
        PrintAsciiCalendar()
    End Sub

    Private Sub SetupChart()

        chartBiorhythm.Series.Clear()
        chartBiorhythm.ChartAreas(0).AxisX.LabelStyle.Format = "MM/dd"
        chartBiorhythm.ChartAreas(0).AxisY.Minimum = -1.2
        chartBiorhythm.ChartAreas(0).AxisY.Maximum = 1.2
        chartBiorhythm.ChartAreas(0).AxisY.Interval = 0.5

        ' Add the three series
        CreateSeries("Physical", Color.Red)
        CreateSeries("Emotional", Color.Blue)
        CreateSeries("Intellectual", Color.Green)

    End Sub

    Private Sub CreateSeries(name As String, color As Color)
        Dim s As New Series(name)
        s.ChartType = SeriesChartType.Line
        s.BorderWidth = 2
        s.Color = color
        s.XValueType = ChartValueType.Date
        chartBiorhythm.Series.Add(s)
    End Sub

    Private Sub AddTargetMarker(target As Date)

        Dim ca = chartBiorhythm.ChartAreas(0)
        Dim strip As New StripLine()

        strip.Interval = 0
        strip.BackColor = Color.FromArgb(60, Color.Black)
        strip.StripWidth = 0.15
        strip.IntervalOffset = target.ToOADate()
        strip.Text = "Selected Date"
        strip.TextAlignment = StringAlignment.Center
        strip.ForeColor = Color.Black
        strip.Font = New Font("Arial", 8, FontStyle.Bold)

        ca.AxisX.StripLines.Add(strip)

    End Sub

    Private Sub BuildMonthCalendar()

        Dim birth As Date = dtpBirthdate.Value

        Dim firstDay As New Date(dtpTargetDate.Value.Year,
                             dtpTargetDate.Value.Month, 1)

        ' Setup the grid
        dgvMonth.Columns.Clear()
        dgvMonth.Rows.Clear()

        For i = 1 To 7
            dgvMonth.Columns.Add("col" & i.ToString(),
            System.Globalization.CultureInfo.CurrentCulture.DateTimeFormat.AbbreviatedDayNames(i Mod 7))
        Next

        dgvMonth.RowTemplate.Height = 32

        ' Find first weekday position
        Dim startCol As Integer = CInt(firstDay.DayOfWeek)

        Dim curRow As Integer = dgvMonth.Rows.Add()
        Dim curCol As Integer = startCol

        Dim d As Date = firstDay
        Dim lastDay As Integer = Date.DaysInMonth(firstDay.Year, firstDay.Month)

        For day = 1 To lastDay

            Dim daysPast As Integer = (d - birth).Days
            Dim P = Math.Sin(2 * Math.PI * daysPast / PHY_CYCLE)
            Dim E = Math.Sin(2 * Math.PI * daysPast / EMO_CYCLE)
            Dim I = Math.Sin(2 * Math.PI * daysPast / INT_CYCLE)

            Dim critP = (Math.Abs(P) < 0.03)
            Dim critE = (Math.Abs(E) < 0.03)
            Dim critI = (Math.Abs(I) < 0.03)

            Dim cellIcon As String = ""
            Dim bg As Color = Color.White

            Dim critCount As Integer = 0
            If critP Then critCount += 1 : cellIcon &= "🔴"
            If critE Then critCount += 1 : cellIcon &= "🔵"
            If critI Then critCount += 1 : cellIcon &= "🟢"

            If critCount = 2 Then cellIcon = "⭐ " & cellIcon
            If critCount = 3 Then
                cellIcon = "💥"
                bg = Color.LightCoral
            End If

            dgvMonth.Rows(curRow).Cells(curCol).Value = day & " " & cellIcon

            dgvMonth.Rows(curRow).Cells(curCol).Style.BackColor = bg

            ' Next day
            curCol += 1
            If curCol > 6 Then
                curRow = dgvMonth.Rows.Add()
                curCol = 0
            End If

            d = d.AddDays(1)
        Next

    End Sub

    Private Sub PrintAsciiCalendar()

        Dim birth As Date = dtpBirthdate.Value
        Dim shown As New Date(dtpTargetDate.Value.Year,
                          dtpTargetDate.Value.Month, 1)

        Dim sb As New System.Text.StringBuilder()

        sb.AppendLine(shown.ToString("MMMM yyyy"))
        sb.AppendLine("Su Mo Tu We Th Fr Sa")

        Dim startCol As Integer = CInt(shown.DayOfWeek)
        sb.Append(New String(" "c, startCol * 3))

        Dim d As Date = shown
        Dim daysInMonth = Date.DaysInMonth(shown.Year, shown.Month)

        For day = 1 To daysInMonth

            Dim daysPast As Integer = (d - birth).Days
            Dim P = Math.Sin(2 * Math.PI * daysPast / PHY_CYCLE)
            Dim E = Math.Sin(2 * Math.PI * daysPast / EMO_CYCLE)
            Dim I = Math.Sin(2 * Math.PI * daysPast / INT_CYCLE)

            Dim critCount As Integer =
            (If(Math.Abs(P) < 0.03, 1, 0)) +
            (If(Math.Abs(E) < 0.03, 1, 0)) +
            (If(Math.Abs(I) < 0.03, 1, 0))

            Dim marker As String =
            If(critCount = 3, "!", If(critCount = 2, "*", If(critCount = 1, "+", " ")))

            sb.Append(day.ToString().PadLeft(2) & marker & " ")

            If (startCol Mod 7) = 6 Then sb.AppendLine()
            startCol += 1

            d = d.AddDays(1)
        Next

        txtASCII.Text = sb.ToString()

    End Sub

End Class

